/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mx_auto_config.h"
#include "myriexpress.h"
#if !MX_OS_WINNT
#include <unistd.h>
#else
#include "mx_uni.h"
#endif
#include "test_common.h"

struct cmddef {
  char *name;
  void (*func)(char **, int);
};

struct variable {
  char name[128];
  union {
    struct {
      mx_endpoint_t ep;
    } ep;
    struct {
      mx_endpoint_addr_t addr;
    } targ;
    struct req_var {
      mx_request_t request;
      mx_endpoint_t ep;
      int type;
      unsigned char *buf;
      uint32_t len;
      int verify;
      int seed;
    } req;
  } u;

  struct variable *next;
  struct variable *prev;
};

/* for type field above */
enum { RQ_SEND, RQ_SSEND, RQ_RECV, RQ_BCAST, RQ_BARR };

#define LIST_CLEAR(L) do { L.next = L.prev = &L; } while (0)
#define LIST_REMOVE(I)                \
  do {                                \
    (I)->prev->next = (I)->next;      \
    (I)->next->prev = (I)->prev;      \
  } while (0)
#define LIST_INSERT(L, I)     \
  do {                        \
    (I)->prev = (L)->prev;    \
    (I)->next = (L);          \
    (L)->prev->next = (I);    \
    (L)->prev = (I);          \
  } while (0)

void do_open(char **, int);
void do_close(char **, int);
void do_sleep(char **, int);
void do_target(char **, int);
void do_send(char **, int);
void do_ssend(char **, int);
void do_recv(char **, int);
void do_test(char **, int);
void do_iprobe(char **, int);

/*
 * Globals
 */

int Verbose = 0;
int Line = 0;
struct cmddef Commands[] = {
  { "open", do_open },
  { "close", do_close },
  { "target", do_target },
  { "send", do_send },
  { "ssend", do_ssend },
  { "recv", do_recv },
  { "test", do_test },
  { "sleep", do_sleep },
  { "iprobe", do_iprobe },
  { NULL, NULL }
};

struct variable Endpoints;
struct variable Targets;
struct variable Requests;

void 
usage()
{
  fprintf(stderr, "Usage: mx_interp [args]\n");
  fprintf(stderr, "-f filename - file with interpreter instructions\n");
  fprintf(stderr, "-v - verbose\n");
  fprintf(stderr, "-h - help\n");
}

void
fatal(char *s)
{
  fprintf(stderr, "Line %d: %s\n", Line, s);
  exit(1);
}

void
fatal_mx(char *s, mx_return_t rc)
{
  char buf[256];

  sprintf(buf, "%s: %s", s, mx_strerror(rc));
  fatal(buf);
}

void
verbose(char *s)
{
  if (Verbose) printf("%d: %s\n", Line, s);
}

#define verbose1(FMT, A)                   \
  do if (Verbose) {                        \
    char buf[256];                         \
    sprintf(buf, FMT, A);                  \
    verbose(buf);                          \
  } while (0)
#define verbose2(FMT, A, B)                \
  do if (Verbose) {                        \
    char buf[256];                         \
    sprintf(buf, FMT, A, B);               \
    verbose(buf);                          \
  } while (0)
#define verbose3(FMT, A, B, C)             \
  do if (Verbose) {                        \
    char buf[256];                         \
    sprintf(buf, FMT, A, B, C);            \
    verbose(buf);                          \
  } while (0)
#define verbose4(FMT, A, B, C, D)          \
  do if (Verbose) {                        \
    char buf[256];                         \
    sprintf(buf, FMT, A, B, C, D);         \
    verbose(buf);                          \
  } while (0)
#define verbose5(FMT, A, B, C, D, E)       \
  do if (Verbose) {                        \
    char buf[256];                         \
    sprintf(buf, FMT, A, B, C, D, E);      \
    verbose(buf);                          \
  } while (0)

uint32_t
number(char *s)
{
  uint32_t val;

  if (s[0] == '0' && s[1] == 'x') {
    sscanf(s+2, "%x", &val);
  } else {
    sscanf(s, "%d", &val);
  }

  return val;
}

/*
 * find a variable name
 */
struct variable *
find_var(struct variable *vroot,
         char *name)
{
  struct variable *v;

  for (v = vroot->next; v != vroot; v = v->next) {
    if (strcmp(v->name, name) == 0) return v;
  }

  return NULL;
}

/*
 * Allocate a record for a new variable.  A variable of matching name must
 * not exist
 */
struct variable *
new_var(struct variable *vroot,
             char *name)
{
  struct variable *v;

  /* make sure name is not used already */
  v = find_var(vroot, name);
  if (v != NULL) {
    fatal("Duplicate variable name");
  }

  v = (struct variable *) calloc(sizeof(*v), 1);
  if (v == NULL) {
    fatal("malloc variable");
  }

  /* fill in the name */
  strncpy(v->name, name, sizeof(v->name));

  /* put it into the list and return */
  LIST_INSERT(vroot, v);
  return v;
}

/*
 * sleep N
 */
enum {
  SLP_SLP=0, SLP_N, SLP_CNT
};
void
do_sleep(
  char **words,
  int nw)
{
  int s;

  /* make sure we have the right # of arguments */
  if (nw != SLP_CNT) {
    fatal("bad # of args for sleep");
  }


  s = number(words[SLP_N]);

  verbose1("Sleep %d seconds", s);

  sleep(s);

  verbose("done sleeping");
}
 
/*
 * target target_ID hostame nic_index endpoint_id filter
 */
enum {
 TARG_TARG=0, TARG_NAME, TARG_EP, TARG_HOST, TARG_NIC, TARG_EID, TARG_FILTER, TARG_CNT
};
void
do_target(
  char **words,
  int nw)
{
  mx_return_t rc;
  uint32_t filter;
  uint32_t eid;
  int nic_index;
  uint64_t nid[8];
  struct variable *v;
  struct variable *ep;

  /* make sure we have the right # of arguments */
  if (nw != TARG_CNT) {
    fatal("bad # of args for target");
  }

  /* get a variable to hold the endpoint */
  v = new_var(&Targets, words[TARG_NAME]);

  /* find the NIC ID */
  nic_index = number(words[TARG_NIC]);
  if (nic_index != 0) {
    fatal("bad NIC index, should be 0");
  }

  rc = mx_hostname_to_nic_id(words[TARG_HOST], nid);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_hostname_to_nic_id", rc);
  }

  verbose1("Found NIC for nic_name %s", words[TARG_HOST]);

  eid = number(words[TARG_EID]);
  filter = number(words[TARG_FILTER]);
  ep = find_var(&Endpoints, words[TARG_EP]);
  rc = mx_connect(ep->u.ep.ep, nid[nic_index], eid, filter, MX_INFINITE, &(v->u.targ.addr));
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_connect", rc);
  }

  verbose4("target %s created: nic=%"PRIx64", eid=%d, filter=%x", v->name,
    nid[nic_index], eid, filter);
}

/*
 * send endpoint name target len match seed
 */
enum {
 TEST_TEST=0, TEST_REQ, TEST_CNT
};
void
do_test(
  char **words,
  int nw)
{
  mx_return_t rc;
  struct variable *vr;
  mx_status_t stat;
  struct req_var *req;
  int i;
  unsigned char check;
  uint32_t result;

  /* make sure we have the right # of arguments */
  if (nw != TEST_CNT) {
    fatal("bad # of args for test");
  }

  /* find our request */
  vr = find_var(&Requests, words[TEST_REQ]);
  if (vr == NULL) {
    char buf[80];
    sprintf(buf, "Unknown request for test: %s", words[TEST_REQ]);
    fatal(buf);
  }
  req = &vr->u.req;

  verbose1("Waiting for request %s", vr->name);

  /* wait for completion */
  while (1) {
    rc = mx_test(req->ep, &req->request, &stat, &result);
    if (rc != MX_SUCCESS) {
      fatal_mx("mx_test", rc);
    }

    if (result == 1) {
      break;
    }
  }

  /* Take some finishing action based on type */
  switch (req->type) {

  case RQ_SEND:
  case RQ_SSEND:
    if (stat.msg_length != req->len) {
      char s[128];
      sprintf(s, "status returned incorrect length for send, %d should be %d\n",
      	stat.msg_length, req->len);
      fatal(s);
    }
    verbose2("%ssend %s complete", (req->type == RQ_SEND ? "" : "s"), vr->name);
    free(req->buf);
    break;

  case RQ_RECV:
    verbose4("recv %s complete, len = %d, match = %x:%x",
      vr->name, stat.xfer_length, MX_U32(stat.match_info),
	     MX_L32(stat.match_info));

    if (req->verify) {
      srand(req->seed);
      for (i=0; i<stat.xfer_length; ++i) {
        check = rand() & 0xFF;
        if (req->buf[i] != check) {
	  char s[128];
	  sprintf(s, "data error: rx %x != %x, pos = %d\n", 
	    req->buf[i], check, i);
	  fatal(s);
	}
      }
    }

    free(req->buf);
    break;

  default:
    fatal("Internal error: unrecognized request type");
    break;
  }

  /* clobber the variable */
  LIST_REMOVE(vr);
  free(vr);
}

/*
 * recv name ep len match0 match1 mask seed
 */
enum {
 RECV_RECV=0, RECV_NAME, RECV_EP, RECV_LEN,
 RECV_MATCH0, RECV_MATCH1, RECV_MASK, RECV_SEED, RECV_CNT
};
void
do_recv(
  char **words,
  int nw)
{
  mx_return_t rc;
  struct variable *vr;
  struct variable *ve;
  uint32_t len;
  unsigned char *buf;
  mx_segment_t seg;
  uint32_t match_data0;
  uint32_t match_data1;
  uint64_t match_info;
  uint64_t match_mask = 0; /* satisfy -Wunused on broken compilers */
  struct req_var *req;

  /* make sure we have the right # of arguments */
  if (nw != RECV_CNT) {
    fatal("bad # of args for recv");
  }

  /* get a record for keeping the request */
  vr = new_var(&Requests, words[RECV_NAME]);
  req = &vr->u.req;
  req->type = RQ_RECV;

  /* find our endpoint */
  ve = find_var(&Endpoints, words[RECV_EP]);
  if (ve == NULL) {
    fatal("Unknown endpoint for recv");
  }
  req->ep = ve->u.ep.ep;  /* associate endpoint with request */

  /* create a buffer to hold data for recv */
  len = number(words[RECV_LEN]);
  buf = (unsigned char *) malloc(len);
  if (buf == NULL) {
    fatal("malloc buffer for recv");
  }
  seg.segment_ptr = buf;
  seg.segment_length = len;

  /* save buffer so we can free it on completion */
  req->buf = buf;

  /* Remember whether we need to verify data or not */
  if (strcmp(words[RECV_SEED], "noseed") != 0) {
    req->verify = 1;
    req->seed = number(words[RECV_SEED]);
  } else {
    req->verify = 0;
  }

  /* get match info */
  match_data0 = number(words[RECV_MATCH0]);
  match_data1 = number(words[RECV_MATCH1]);
  match_info = (uint64_t)match_data0 << 32 | match_data1;
  if (strcmp(words[RECV_MASK], "none") == 0) {
    match_mask = MX_MATCH_MASK_NONE;
  } else if (strcmp(words[RECV_MASK], "b") == 0) {
    match_mask = UINT64_C(0xffffffff0000ffff);
  } else if (strcmp(words[RECV_MASK], "c") == 0) {
    match_mask = UINT64_C(0xffffffffffff0000);
  } else if (strcmp(words[RECV_MASK], "bc") == 0) {
    match_mask = UINT64_C(0xffffffff00000000);
  } else {
    fatal("unrecognized mask value: none, b, c, or bc");
  }


  /* ready to post the recv */
  rc = mx_irecv(req->ep, &seg, 1, match_info, match_mask, NULL,
  	&req->request);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_irecv", rc);
  }

  verbose5("recv %s posted, len = %d, match = %x:%x, mask=%s",
      vr->name, len, match_data0, match_data1, words[RECV_MASK]);
}

/*
 * iprobe ep match0 match1 mask yesno len
 */
enum {
 IPROBE_IPROBE=0, IPROBE_EP,
 IPROBE_MATCH0, IPROBE_MATCH1, IPROBE_MASK, IPROBE_YESNO, IPROBE_LEN,
 IPROBE_CNT
};
void
do_iprobe(
  char **words,
  int nw)
{
  mx_return_t rc;
  struct variable *ve;
  uint32_t match_data0;
  uint32_t match_data1;
  uint64_t match_info;
  uint64_t match_mask  = 0; /* satisfy -Wunused on broken compilers */
  uint32_t yesno;
  int32_t len;
  mx_status_t stat;
  uint32_t result;

  /* make sure we have the right # of arguments */
  if (nw != IPROBE_CNT) {
    fatal("bad # of args for iprobe");
  }

  /* find our endpoint */
  ve = find_var(&Endpoints, words[IPROBE_EP]);
  if (ve == NULL) {
    fatal("Unknown endpoint for iprobe");
  }

  /* get match info */
  match_data0 = number(words[IPROBE_MATCH0]);
  match_data1 = number(words[IPROBE_MATCH1]);
  match_info = (uint64_t)match_data0 << 32 | match_data1;
  if (strcmp(words[IPROBE_MASK], "none") == 0) {
    match_mask = MX_MATCH_MASK_NONE;
  } else if (strcmp(words[IPROBE_MASK], "b") == 0) {
    match_mask = UINT64_C(0xffffffff0000ffff);
  } else if (strcmp(words[IPROBE_MASK], "c") == 0) {
    match_mask = UINT64_C(0xffffffffffff0000);
  } else if (strcmp(words[IPROBE_MASK], "bc") == 0) {
    match_mask = UINT64_C(0xffffffff00000000);
  } else {
    fatal("unrecognized mask value: none, b, c, or bc");
  }

  /* learn whether we should succeed or not */
  yesno = (strcmp(words[IPROBE_YESNO], "yes") == 0) ? 1 : 0;
  len = number(words[IPROBE_LEN]);

  /* ready to post the iprobe */
  rc = mx_iprobe(ve->u.ep.ep, match_info, match_mask, &stat, &result);
  if (rc != MX_SUCCESS) {
    fatal_mx("iprobe 2", rc);
  }

  if (result == 1) {

    verbose4("iprobe found, len = %d, match = %x:%x, mask=%s",
	     len, match_data0, match_data1, words[IPROBE_MASK]);

    if (!yesno) {
      fatal("iprobe succeeded where it should have failed");
    }
    if (len >= 0 && stat.msg_length != len) {
      char buf[80];
      sprintf(buf, "iprobe length was %d, should be %d",
      		stat.msg_length, len);
      fatal(buf);
    }
  } else if (result == 0) {

    verbose4("iprobe notfound, len = %d, match = %x:%x, mask=%s",
      len, match_data0, match_data1, words[IPROBE_MASK]);
      
    if (yesno) {
      fatal("iprobe failed where it should have succeeded");
    }
  } else {
    char buf[80];
    sprintf(buf, "iprobe returned bad result = %d", result);
    fatal(buf);
  }

}

/*
 * send name endpoint target len match0 match1 seed
 */
enum {
 SEND_SEND=0, SEND_NAME, SEND_EP, SEND_TARG, SEND_LEN,
 SEND_MATCH0, SEND_MATCH1, SEND_SEED, SEND_CNT
};
void
do_send(
  char **words,
  int nw)
{
  mx_return_t rc;
  struct variable *vs;
  struct req_var *req;
  struct variable *vt;
  struct variable *ve;
  uint32_t len;
  unsigned char *buf;
  mx_segment_t seg;
  uint32_t match0;
  uint32_t match1;
  uint64_t match_info;
  int i;

  /* make sure we have the right # of arguments */
  if (nw != SEND_CNT) {
    fatal("bad # of args for send");
  }

  /* get a record for keeping the request */
  vs = new_var(&Requests, words[SEND_NAME]);
  req = &vs->u.req;
  req->type = RQ_SEND;

  /* find our endpoint */
  ve = find_var(&Endpoints, words[SEND_EP]);
  if (ve == NULL) {
    fatal("Unknown endpoint for send");
  }
  req->ep = ve->u.ep.ep;  /* associate endpoint with request */

  /* find our target */
  vt = find_var(&Targets, words[SEND_TARG]);
  if (vt == NULL) {
    fatal("Unknown target for send");
  }

  /* create a buffer of data to send */
  len = number(words[SEND_LEN]);
  buf = (unsigned char *) malloc(len);
  if (buf == NULL) {
    fatal("malloc buffer for send");
  }
  seg.segment_ptr = buf;
  seg.segment_length = len;

  /* save buffer so we can free it on completion */
  req->buf = buf;
  req->len = len;

  /* Fill in with data unless "noseed" specified */
  if (strcmp(words[SEND_SEED], "noseed") != 0) {
    srand(number(words[SEND_SEED]));
    for (i=0; i<len; ++i) buf[i] = rand() & 0xFF;
  }

  /* get match info */
  match0 = number(words[SEND_MATCH0]);
  match1 = number(words[SEND_MATCH1]);
  match_info = (uint64_t)match0 << 32 | match1;

  /* ready to post the send */
  rc = mx_isend(req->ep, &seg, 1, vt->u.targ.addr, match_info, NULL,
  	&req->request);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_isend", rc);
  }

  verbose5("send %s posted, target = %s, len = %d, match = %x:%x",
  	vs->name, words[SEND_TARG], len,
  	match0, match1);
}

/*
 * ssend name endpoint target len match0 match1 seed
 */
enum {
 SSEND_SSEND=0, SSEND_NAME, SSEND_EP, SSEND_TARG, SSEND_LEN,
 SSEND_MATCH0, SSEND_MATCH1, SSEND_SEED, SSEND_CNT
};
void
do_ssend(
  char **words,
  int nw)
{
  mx_return_t rc;
  struct variable *vs;
  struct req_var *req;
  struct variable *vt;
  struct variable *ve;
  uint32_t len;
  unsigned char *buf;
  mx_segment_t seg;
  uint32_t match0;
  uint32_t match1;
  uint64_t match_info;
  int i;

  /* make sure we have the right # of arguments */
  if (nw != SSEND_CNT) {
    fatal("bad # of args for ssend");
  }

  /* get a record for keeping the request */
  vs = new_var(&Requests, words[SSEND_NAME]);
  req = &vs->u.req;
  req->type = RQ_SSEND;

  /* find our endpoint */
  ve = find_var(&Endpoints, words[SSEND_EP]);
  if (ve == NULL) {
    fatal("Unknown endpoint for ssend");
  }
  req->ep = ve->u.ep.ep;  /* associate endpoint with request */

  /* find our target */
  vt = find_var(&Targets, words[SSEND_TARG]);
  if (vt == NULL) {
    fatal("Unknown target for ssend");
  }

  /* create a buffer of data to send */
  len = number(words[SSEND_LEN]);
  buf = (unsigned char *) malloc(len);
  if (buf == NULL) {
    fatal("malloc buffer for ssend");
  }
  seg.segment_ptr = buf;
  seg.segment_length = len;

  /* save buffer so we can free it on completion */
  req->buf = buf;
  req->len = len;

  /* Fill in with data unless "noseed" specified */
  if (strcmp(words[SSEND_SEED], "noseed") != 0) {
    srand(number(words[SSEND_SEED]));
    for (i=0; i<len; ++i) buf[i] = rand() & 0xFF;
  }

  /* get match info */
  match0 = number(words[SSEND_MATCH0]);
  match1 = number(words[SSEND_MATCH1]);
  match_info = (uint64_t)match0 << 32 | match1;

  /* ready to post the ssend */
  rc = mx_issend(req->ep, &seg, 1, vt->u.targ.addr, match_info, NULL,
  	&req->request);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_issend", rc);
  }

  verbose5("ssend %s posted, target = %s, len = %d, match = %x:%x",
  	vs->name, words[SSEND_TARG], len,
  	match0, match1);
}

/*
 * open name nic endpoint_id filter
 */
enum {
 OPEN_OPEN=0, OPEN_NAME, OPEN_NID, OPEN_EID, OPEN_FILTER, OPEN_CNT
};
void
do_open(
  char **words,
  int nw)
{
  mx_return_t rc;
  uint32_t filter;
  uint32_t eid;
  uint32_t got_eid;
  uint32_t bnum;
  uint64_t got_nid;
  struct variable *v;
  mx_endpoint_addr_t addr;

  /* make sure we have the right # of arguments */
  if (nw != OPEN_CNT) {
    fatal("bad # of args for open");
  }

  /* get a record for keeping the endpoint */
  v = new_var(&Endpoints, words[OPEN_NAME]);

  /* get args */
  if (strcmp(words[OPEN_NID], "any") == 0) {
    bnum = MX_ANY_NIC;
  } else {
    fprintf(stderr, "Cannot specify NIC at the moment...\n");
    exit(1);
    /*
    bnum = number(words[OPEN_NID]);
    */
  }

  if (strcmp(words[OPEN_EID], "any") == 0) {
    eid = MX_ANY_ENDPOINT;
  } else {
    eid = number(words[OPEN_EID]);
  }

  filter = number(words[OPEN_FILTER]);

  rc = mx_open_endpoint(bnum, eid, filter, NULL, 0, &v->u.ep.ep);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_open_endpoint", rc);
  }

  rc = mx_get_endpoint_addr(v->u.ep.ep, &addr);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_get_endpoint_addr", rc);
  }

  rc = mx_decompose_endpoint_addr(addr, &got_nid, &got_eid);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_decompose_endpoint_addr", rc);
  }

  verbose3("Endpoint \"%s\" opened: nid=0x%"PRIx64", eid=%d", v->name,
  	got_nid, got_eid);
}

/*
 * close name
 */
enum {
 CLS_CLS=0, CLS_NAME, CLS_CNT
};
void
do_close(
  char **words,
  int nw)
{
  mx_return_t rc;
  struct variable *v;

  /* make sure we have the right # of arguments */
  if (nw != CLS_CNT) {
    fatal("bad # of args for close");
  }

  /* find this endpoint */
  v = find_var(&Endpoints, words[OPEN_NAME]);
  if (v == NULL) {
    fatal("Cannot find endpoint to close");
  }

  rc = mx_close_endpoint(v->u.ep.ep);
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_close_endpoint", rc);
  }

  verbose1("Endpoint \"%s\" closed", v->name);

  /* remove this from list of endpoints */
  LIST_REMOVE(v);
  free(v);
}
  

int
any(
	char *s,
	int c)
{
  while (*s) if (*s++ == c) return 1;
  return 0;
}

/*
 * read a line into words
 */
char **
line2words(
	char *buf,
	char *seps,
	int  wmax,
	int *wcnt)
{
  char  *wp;
  int    wc;
  char **words;

  if (wmax < 1 || wmax > BUFSIZ) wmax = BUFSIZ;

  words = (char **) malloc(sizeof(char *) * BUFSIZ);
  if (words == NULL) {
    perror("malloc");
    exit(1);
  }

  wc = 0;
  wp = buf;
  while (wc < wmax) {
    while (any(seps, *wp)) ++wp;
    if (*wp == '\0') break;

    words[wc++] = wp;
    while (!any(seps, *wp) && *wp != '\0') ++wp;
    if (wc < wmax && *wp != '\0') *wp++ = '\0';
  }

  if (wcnt) *wcnt = wc;
  return words;
}

void
init()
{
  mx_return_t rc;

  mx_set_error_handler(MX_ERRORS_RETURN);
  rc = mx_init();
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_init", rc);
  }

  LIST_CLEAR(Endpoints);
  LIST_CLEAR(Targets);
  LIST_CLEAR(Requests);
}

void
finalize()
{
  mx_return_t rc;

  rc = mx_finalize();
  if (rc != MX_SUCCESS) {
    fatal_mx("mx_finalize", rc);
  }
}

int
main(int argc, char **argv)
{
  int c;
  extern char *optarg;
  FILE *in;
  char buf[512];
  char **words;
  int nw;
  int i;

  in = stdin;

  while ((c = getopt(argc, argv, "hvf:")) != EOF) switch (c) {
  case 'v':
    Verbose = 1;
    break;
  case 'f':
    in = fopen(optarg, "r");
    if (in == NULL) {
      perror(optarg);
      exit(1);
    }
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  /* initialize stuff */
  init();

  /* loop on commands */
  Line = 0;
  while (fgets(buf, sizeof(buf), in) != NULL) {
    ++Line;

    /* break into words, TAB and SPACE are legal seperators */
    words = line2words(buf, "	 \n", 0, &nw);

    if (nw == 0 || words[0][0] == '#') continue;

    for (i=0; Commands[i].name != NULL; ++i) {
      if (strcmp(Commands[i].name, words[0]) == 0) {
        Commands[i].func(words, nw);
	break;
      }
    }

    /* make sure we liked the command */
    if (Commands[i].name == NULL) {
      fatal("Unrecognized command");
    }

    free(words);
  }

  /* close up and exit */
  finalize();
  exit(0);
}
